<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\PostsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\Posts\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\PostsImport;
use App\Models\Post;
use App\Models\Therapist;
use App\Models\BlogCategory;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

class TherapistPostsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_blog');
        $this->middleware('permission:therapist.add_blog', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_blog', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_blog', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_blog', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_blog', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_posts =  $therapist->therapist_posts();
            if ($req->trash && $req->trash == 'with') {
                $therapist_posts =  $therapist_posts->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_posts =  $therapist_posts->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_posts = $therapist_posts->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_posts = $therapist_posts->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_posts = $therapist_posts->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_posts = $therapist_posts->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_posts = $therapist_posts->get();
                return $therapist_posts;
            }
            $therapist_posts = $therapist_posts->get();
            return $therapist_posts;
        }
        $therapist_posts = $therapist->therapist_posts()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_posts;
    }


    /*********View All Posts  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_posts = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_posts.index', compact('therapist_posts', 'therapist'));
    }

    /*********View Create Form of Post  ***********/
    public function create(Therapist $therapist)
    {
        $blog_categories = BlogCategory::active()->get();
        $tags = Tag::active()->get();

        return view('super_admins.therapists.therapist_posts.create', compact('blog_categories', 'therapist', 'tags'));
    }

    /*********Store Post  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {

        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['created_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            $data['image'] = uploadCroppedFile($request, 'image', 'therapist_posts');
            $therapist_post = $therapist->therapist_posts()->create($data);
            $therapist_post->slug = Str::slug($therapist_post->name . ' ' . $therapist_post->id, '-');
            $therapist_post->save();
            $therapist_post = $therapist->therapist_posts()->withAll()->find($therapist_post->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_posts.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_posts.index', $therapist->id)->with('message', 'Post Created Successfully')->with('message_type', 'success');
    }

    /*********View Post  ***********/
    public function show(Therapist $therapist, Post $therapist_post)
    {
        if ($therapist->id != $therapist_post->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_posts.show', compact('therapist_post', 'therapist'));
    }

    /*********View Edit Form of Post  ***********/
    public function edit(Therapist $therapist, Post $therapist_post)
    {
        if ($therapist->id != $therapist_post->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $blog_categories = BlogCategory::active()->get();
        $tags = Tag::active()->get();
        return view('super_admins.therapists.therapist_posts.edit', compact('therapist_post', 'blog_categories', 'therapist', 'tags'));
    }

    /*********Update Post  ***********/
    public function update(CreateRequest $request, Therapist $therapist, Post $therapist_post)
    {
        if ($therapist->id != $therapist_post->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $request->merge(['last_updated_by_user_id' => auth()->user()->id]);
            $data = $request->all();
            if ($request->image) {
                $data['image'] = uploadCroppedFile($request, 'image', 'therapist_posts', $therapist_post->image);
            } else {
                $data['image'] = $therapist_post->image;
            }
            $therapist_post->update($data);
            $therapist_post = Post::find($therapist_post->id);
            $slug = Str::slug($therapist_post->name . ' ' . $therapist_post->id, '-');
            $therapist_post->update([
                'slug' => $slug
            ]);
            $therapist_post->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_posts.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_posts.index', $therapist->id)->with('message', 'Post Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_posts = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_posts." . $extension;
        return Excel::download(new PostsExport($therapist_posts), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new PostsImport, $file);
        return redirect()->back()->with('message', 'Post Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Post ***********/
    public function destroy(Therapist $therapist, Post $therapist_post)
    {
        if ($therapist->id != $therapist_post->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $therapist_post->delete();
        return redirect()->back()->with('message', 'Post Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Post ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_post)
    {
        if ($therapist->id != $therapist_post->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $therapist_post = Post::withTrashed()->find($therapist_post);
        if ($therapist_post) {
            if ($therapist_post->trashed()) {
                if ($therapist_post->image && file_exists(public_path($therapist_post->image))) {
                    unlink(public_path($therapist_post->image));
                }
                $therapist_post->forceDelete();
                return redirect()->back()->with('message', 'Post Category Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Post Category is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Post Category Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Post***********/
    public function restore(Request $request, Therapist $therapist, $therapist_post)
    {
        if ($therapist->id != $therapist_post->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $therapist_post = Post::withTrashed()->find($therapist_post);
        if ($therapist_post->trashed()) {
            $therapist_post->restore();
            return redirect()->back()->with('message', 'Post Category Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Post Category Not Found')->with('message_type', 'error');
        }
    }
}
